/***************************************************************************
*
* Copyright 2010, 2011 BMW Car IT GmbH
* Copyright (C) 2011 DENSO CORPORATION and Robert Bosch Car Multimedia Gmbh
*
*
* Licensed under the Apache License, Version 2.0 (the "License");
* you may not use this file except in compliance with the License.
* You may obtain a copy of the License at
*
*        http://www.apache.org/licenses/LICENSE-2.0
*
* Unless required by applicable law or agreed to in writing, software
* distributed under the License is distributed on an "AS IS" BASIS,
* WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
* See the License for the specific language governing permissions and
* limitations under the License.
*
*
* THE COPYRIGHT HOLDERS DISCLAIM ALL WARRANTIES WITH REGARD TO THIS
* SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND
* FITNESS, IN NO EVENT SHALL THE COPYRIGHT HOLDERS BE LIABLE FOR ANY
* SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER
* RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF
* CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN
* CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
*
****************************************************************************/
#include <time.h>
#include <sys/time.h>
#include <sys/wait.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "WindowSystems/WaylandInputDevice.h"
#include "WindowSystems/wayland-1.0-server-imports.h"

/////////////////////////////////////////////////////////////////////////////

static struct wl_resource*
findResourceForClient(struct wl_list *list, struct wl_client *client)
{
    LOG_DEBUG("WaylandInputDevice", "list=" << list << ", client=" << client);
    struct wl_resource *r;
    wl_list_for_each(r, list, link)
    {
        if (r->client == client)
            return r;
    }
    return NULL;
}

/////////////////////////////////////////////////////////////////////////////

WaylandInputDevice::WaylandInputDevice(struct wl_display *display)
: m_wlDisplay(display)
, m_hasPointer(false)
, m_hasKeyboard(false)
, m_hasTouch(false)
, m_wlTouchFocus(NULL)
, m_wlTouchFocusResource(NULL)
, m_pInpManager(NULL)
{
    LOG_DEBUG("WaylandInputDevice", "display=" << display);
    wl_seat_init(&m_wlSeat);
    strcpy(m_nameOfSeat, "seat0");

    wl_display_add_global(m_wlDisplay,
                            &wl_seat_interface,
                            this,
                            WaylandInputDevice::bindInputDevice);
}

WaylandInputDevice::WaylandInputDevice(struct wl_display *display, char *pSeatName, InputManager *pInpManager)
: m_wlDisplay(display)
, m_hasPointer(false)
, m_hasKeyboard(false)
, m_hasTouch(false)
, m_wlTouchFocus(NULL)
, m_wlTouchFocusResource(NULL)
, m_pInpManager(pInpManager)
{
    wl_seat_init(&m_wlSeat);
    if (NULL != pSeatName)
    {
        strncpy(m_nameOfSeat, pSeatName, (MAX_SEATNAME_LNGTH - 2));
        m_nameOfSeat[MAX_SEATNAME_LNGTH - 1] = '\0';
    }
    else
    {
        strcpy(m_nameOfSeat, "seat0");
    }
    std::string seatName(m_nameOfSeat);
    LOG_DEBUG("WaylandInputDevice", "display=" << display << ", seatName=" << seatName);
    wl_display_add_global(m_wlDisplay,
                            &wl_seat_interface,
                            this,
                            WaylandInputDevice::bindInputDevice);
}

WaylandInputDevice::~WaylandInputDevice()
{
    releaseDevices();
}

void
WaylandInputDevice::initPointerDevice()
{
    if (m_hasPointer)
        return;

    wl_pointer_init(&m_deviceInterfaces.wlPointer);
    wl_seat_set_pointer(&m_wlSeat, &m_deviceInterfaces.wlPointer);

    m_hasPointer = true;
    LOG_INFO("WaylandInputDevice", "Available pointer");
}

void
WaylandInputDevice::initKeyboardDevice()
{
    if (m_hasKeyboard)
        return;

    wl_keyboard_init(&m_deviceInterfaces.wlKeyboard);
    wl_seat_set_keyboard(&m_wlSeat, &m_deviceInterfaces.wlKeyboard);

    m_hasKeyboard = true;
    LOG_INFO("WaylandInputDevice", "Available keyboard");
}

void
WaylandInputDevice::initTouchDevice()
{
    if (m_hasTouch)
        return;

    wl_touch_init(&m_deviceInterfaces.wlTouch);
    wl_seat_set_touch(&m_wlSeat, &m_deviceInterfaces.wlTouch);

    m_hasTouch = true;
    LOG_INFO("WaylandInputDevice", "Available touch");
}

void
WaylandInputDevice::releaseDevices()
{
    /* In Wayland 1.2, wl_pointer_release, wl_keyboard_release and
       wl_touch_release are now handled by the client */
    /*
    if (m_hasPointer)
        wl_pointer_release(&m_deviceInterfaces.wlPointer);

    if (m_hasKeyboard)
        wl_keyboard_release(&m_deviceInterfaces.wlKeyboard);

    if (m_hasTouch)
        wl_touch_release(&m_deviceInterfaces.wlTouch);
    */
}

void
WaylandInputDevice::cleanupDataDeviceForClient(struct wl_client *client, bool destroyDev)
{
    (void)client;
    (void)destroyDev;
}

void
WaylandInputDevice::bindInputDevice(struct wl_client *client,
                                    void *data,
                                    uint32_t version,
                                    uint32_t id)
{
    LOG_DEBUG("WaylandInputDevice",
              "client=" << client << ", data=" << data << ", ID=" << id);
    ilmInputDevice devBitMask = 0;

    struct wl_resource *resource =
        wl_client_add_object(client,
                             &wl_seat_interface,
                             &WaylandInputDevice::m_seatInterface,
                             id,
                             data);

    WaylandInputDevice *inputDevice = static_cast<WaylandInputDevice*>(data);
    struct wl_seat *seat = inputDevice->seat();

    resource->destroy = WaylandInputDevice::destroyResource;
    wl_list_insert(&seat->base_resource_list, &resource->link);

    uint32_t caps = 0;

    if (inputDevice->hasPointer())
    {
        caps |= WL_SEAT_CAPABILITY_POINTER;
        devBitMask |= ILM_INPUT_DEVICE_POINTER;
        LOG_DEBUG("WaylandInputDevice",
                  "Device, ID=" << id << " has pointer capability");
    }
    if (inputDevice->hasKeyboard())
    {
        caps |= WL_SEAT_CAPABILITY_KEYBOARD;
        devBitMask |= ILM_INPUT_DEVICE_KEYBOARD;
        LOG_DEBUG("WaylandInputDevice",
                  "Device, ID=" << id << " has keyboard capability");
    }
    if (inputDevice->hasTouch())
    {
        caps |= WL_SEAT_CAPABILITY_TOUCH;
        devBitMask |= ILM_INPUT_DEVICE_TOUCH;
        LOG_DEBUG("WaylandInputDevice",
                  "Device, ID=" << id << " has touch capability");
    }

    if (version >= WL_SEAT_NAME_SINCE_VERSION)
    {
        wl_seat_send_name(resource, inputDevice->m_nameOfSeat);
    }
    wl_seat_send_capabilities(resource, caps);
}

void
WaylandInputDevice::destroyResource(struct wl_resource *resource)
{
    WaylandInputDevice *inputDevice = static_cast<WaylandInputDevice*>(resource->data);
    if (inputDevice->keyboardDevice()->focus_resource == resource)
    {
        LOG_DEBUG("WaylandInputDevice",
                  "Destroy keyboard, resource=" << resource);
        inputDevice->keyboardDevice()->focus_resource = 0;
    }
    if (inputDevice->pointerDevice()->focus_resource == resource)
    {
        LOG_DEBUG("WaylandInputDevice",
                  "Destroy pointer, resource=" << resource);
        inputDevice->pointerDevice()->focus_resource = 0;
    }

    inputDevice->cleanupDataDeviceForClient(resource->client, true);

    wl_list_remove(&resource->link);
}

void
WaylandInputDevice::destroyDeviceResource(struct wl_resource *resource)
{
    LOG_DEBUG("WaylandInputDevice", "resource=" << resource);
    wl_list_remove(&resource->link);
}

void
WaylandInputDevice::pointerRelease(struct wl_client *client, struct wl_resource *resource)
{
    LOG_DEBUG("WaylandInputDevice",
              "client=" << client << ", resource=" << resource);

    wl_resource_destroy(resource);
}

const struct wl_pointer_interface WaylandInputDevice::m_pointerInterface = {
    setCursor,
    pointerRelease
};

void
WaylandInputDevice::setCursor(wl_client *client,
                              wl_resource *resource,
                              uint32_t serial, wl_resource *surface_resource,
                              int32_t x, int32_t y)
{
    (void) surface_resource;

    LOG_DEBUG("WaylandInputDevice",
              "client=" << client << ", resource=" << resource << ", "
              "serial= " << serial << ", x= " << x << ", y= " << y);
}

const struct wl_seat_interface WaylandInputDevice::m_seatInterface = {
    getPointer,
    getKeyboard,
    getTouch,
    seatRelease
};

void
WaylandInputDevice::getPointer(struct wl_client *client,
                               struct wl_resource *resource,
                               uint32_t id)
{
    WaylandInputDevice *inputDevice = static_cast<WaylandInputDevice*>(resource->data);
    wl_pointer *pointer = inputDevice->pointerDevice();
    wl_resource *clientResource = wl_client_add_object(client,
                                                       &wl_pointer_interface,
                                                       &WaylandInputDevice::m_pointerInterface,
                                                       id,
                                                       pointer);
    wl_list_insert(&pointer->resource_list, &clientResource->link);
    clientResource->destroy = WaylandInputDevice::destroyDeviceResource;

    LOG_DEBUG("WaylandInputDevice",
              "client=" << client << ", resource=" << resource << ", "
              "ID=" << id << ", input device=" << inputDevice << ", "
              "pointer=" << pointer << ", client resource=" << clientResource);
}

const struct wl_keyboard_interface WaylandInputDevice::m_keyboardInterface = {
    keyboardRelease
};

void
WaylandInputDevice::keyboardRelease(struct wl_client *client, struct wl_resource *resource)
{
    LOG_DEBUG("WaylandInputDevice",
              "client=" << client << ", resource=" << resource);

    wl_resource_destroy(resource);
}

void
WaylandInputDevice::getKeyboard(struct wl_client *client,
                                struct wl_resource *resource,
                                uint32_t id)
{
    WaylandInputDevice *inputDevice = static_cast<WaylandInputDevice*>(resource->data);
    wl_keyboard *keyboard = inputDevice->keyboardDevice();
    wl_resource *clientResource = wl_client_add_object(client,
                                                       &wl_keyboard_interface,
                                                       &WaylandInputDevice::m_keyboardInterface,
                                                       id,
                                                       keyboard);
    wl_list_insert(&keyboard->resource_list, &clientResource->link);
    clientResource->destroy = WaylandInputDevice::destroyDeviceResource;

    LOG_DEBUG("WaylandInputDevice",
              "client=" << client << ", resource=" << resource << ", "
              "ID=" << id << ", input device=" << inputDevice << ", "
              "keyboard=" << keyboard << ", client resource=" << clientResource);
}

const struct wl_touch_interface WaylandInputDevice::m_touchInterface = {
    touchRelease
};

void
WaylandInputDevice::touchRelease(struct wl_client *client, struct wl_resource *resource)
{
    LOG_DEBUG("WaylandInputDevice",
              "client=" << client << ", resource=" << resource);

    wl_resource_destroy(resource);
}

void
WaylandInputDevice::getTouch(struct wl_client *client,
                             struct wl_resource *resource,
                             uint32_t id)
{
    WaylandInputDevice *inputDevice = static_cast<WaylandInputDevice*>(resource->data);
    wl_touch *touch = inputDevice->touchDevice();
    wl_resource *clientResource = wl_client_add_object(client,
                                                       &wl_touch_interface,
                                                       &WaylandInputDevice::m_touchInterface,
                                                       id,
                                                       touch);
    wl_list_insert(&touch->resource_list, &clientResource->link);
    clientResource->destroy = WaylandInputDevice::destroyDeviceResource;

    LOG_DEBUG("WaylandInputDevice",
              "touch device=" << touch << ", "
              "client=" << client << ", resource=" << resource << ", "
              "ID=" << id << ", input device=" << inputDevice << ", "
              "touch=" << touch << ", client resource=" << clientResource);
}

void
WaylandInputDevice::seatRelease(struct wl_client *client, struct wl_resource *resource)
{
    (void)client;
    wl_resource_destroy(resource);
}

void
WaylandInputDevice::sendAxisEvent(struct wl_surface *surface,
                                  uint32_t axis,
                                  wl_fixed_t value,
                                  uint32_t time)
{
    wl_pointer* pointer = pointerDevice();
    pointer->grab->interface->focus(pointer->grab,
                                        surface,
                                        wl_fixed_from_double(0),
                                        wl_fixed_from_double(0));
    if (pointer->focus_resource)
    {
        LOG_DEBUG("WaylandInputDevice",
                  "pointer device=" << pointer << ", "
                  "native surface=" << surface << ", axis=" << axis << ", "
                  "value=" << value << ", time=" << time);

        wl_pointer_send_axis(pointer->focus_resource,
                             time,
                             axis,
                             value);
    }
    else
    {
        LOG_DEBUG("WaylandInputDevice",
                  "pointer device=" << pointer << ", "
                  "Focus resource not set, native surface=" << surface << ", "
                  "axis=" << axis << ", value=" << value << ", time=" << time);
    }
}

void
WaylandInputDevice::sendMousePressEvent(struct wl_surface *surface,
                                        const Point& globalPos,
                                        const Point& localPos,
                                        uint32_t button,
                                        uint32_t time)
{
    wl_pointer *pointer = pointerDevice();
    /*mouse focus will be only set if we have a button count 0
     * the button count is also tracked in WaylandEventInputEvent.cpp*/
    pointer->button_count--;
    setMouseFocus(surface, globalPos, localPos);
    pointer->button_count++;
    sendMouseMotionEvent(globalPos, localPos, time);
    pointer->button_count++;
    const struct wl_pointer_grab_interface *interface = pointer->grab->interface;
    interface->button(pointer->grab,
                      time,
                      button,
                      WL_POINTER_BUTTON_STATE_PRESSED);
    LOG_DEBUG("WaylandInputDevice",
              "pointer device=" << pointer << ", "
              "native surface=" << surface << ", "
              "global position: x=" << globalPos.x << ", y=" << globalPos.y
              << ", local position: x=" << localPos.x << ", y=" << localPos.y
              << ", button=" << button << ", time=" << time);
}

void
WaylandInputDevice::sendMouseReleaseEvent(const Point& globalPos,
                                          const Point& localPos,
                                          uint32_t button,
                                          uint32_t time)
{
    wl_pointer *pointer = pointerDevice();
    /*if button_count is 0, the surface on which focus resided
     * would have been destroyed.*/
    if (pointer->button_count > 0)
    {
        pointer->button_count--;
        sendMouseMotionEvent(globalPos, localPos, time);
        const struct wl_pointer_grab_interface *interface = pointer->grab->interface;
        interface->button(pointer->grab,
                          time,
                          button,
                          WL_POINTER_BUTTON_STATE_RELEASED);
        LOG_DEBUG("WaylandInputDevice",
                  "pointer device=" << pointer << ", "
                  "global position: x=" << globalPos.x << ", y=" << globalPos.y
                  << ", local position: x=" << localPos.x << ", y=" << localPos.y
                  << ", button=" << button << ", time=" << time);
        return;
    }
    LOG_WARNING("WaylandInputDevice",
              "Ignoring release button for pointer device=" << pointer << ", "
              "global position: x=" << globalPos.x << ", y=" << globalPos.y
              << ", local position: x=" << localPos.x << ", y=" << localPos.y
              << ", button=" << button << ", time=" << time);
}

void
WaylandInputDevice::sendMouseMotionEvent(struct wl_surface* surface,
                                         const Point& globalPos,
                                         const Point& localPos,
                                         uint32_t time)
{
    setMouseFocus(surface, globalPos, localPos);
    sendMouseMotionEvent(globalPos, localPos, time);

    LOG_DEBUG("WaylandInputDevice",
              "native surface=" << surface << ", "
              "global position: x=" << globalPos.x << ", y=" << globalPos.y
              << ", local position: x=" << localPos.x << ", y=" << localPos.y
              << ", time=" << time);
}

void
WaylandInputDevice::sendMouseMotionEvent(const Point& globalPos,
                                         const Point& localPos,
                                         uint32_t time)
{
    wl_pointer *pointer = pointerDevice();
    const struct wl_pointer_grab_interface *interface = pointer->grab->interface;
    pointer->x = wl_fixed_from_double(globalPos.x);
    pointer->y = wl_fixed_from_double(globalPos.y);
    interface->motion(pointer->grab,
                      time,
                      wl_fixed_from_double(localPos.x),
                      wl_fixed_from_double(localPos.y));

    LOG_DEBUG("WaylandInputDevice",
              "pointer device=" << pointer << ", "
              "global position: x=" << globalPos.x << ", y=" << globalPos.y
              << ", local position: x=" << localPos.x << ", y=" << localPos.y
              << ", time=" << time);
}

void
WaylandInputDevice::sendKeyPressEvent(struct wl_surface* surface,
                                      uint32_t time, uint32_t code)
{
    wl_keyboard *keyboard = keyboardDevice();
    if (!keyboard->focus || keyboard->focus != surface)
    {
        setKeyboardFocus(surface);
    }
    if (keyboard->focus_resource)
    {
        uint32_t serial = wl_display_next_serial(m_wlDisplay);
        wl_keyboard_send_key(keyboard->focus_resource,
                             serial, time, code, 1);

        LOG_INFO("WaylandInputDevice",
                  "keyboard=" << keyboard << ", "
                  "native surface=" << surface << ", "
                  "time=" << time << ", code=" << code);
    }
    else
    {
        LOG_INFO("WaylandInputDevice", "No focus resource set, "
                  "keyboard=" << keyboard << ", "
                  "native surface=" << surface << ", "
                  "time=" << time << ", code=" << code);
    }

}

void
WaylandInputDevice::sendKeyReleaseEvent(struct wl_surface* /*surface*/,
                                        uint32_t time, uint32_t code)
{
    wl_keyboard *keyboard = keyboardDevice();
    if (keyboard->focus_resource)
    {
        uint32_t serial = wl_display_next_serial(m_wlDisplay);
        wl_keyboard_send_key(keyboard->focus_resource,
                            serial, time, code, 0);

        LOG_DEBUG("WaylandInputDevice", "keyboard=" << keyboard << ", "
                  "time=" << time << ", code=" << code);
    }
    else
    {
        LOG_DEBUG("WaylandInputDevice", "No focus resource set, "
                  "keyboard=" << keyboard << ", "
                  "time=" << time << ", code=" << code);
    }
}

void
WaylandInputDevice::sendTouchPointEvent(struct wl_surface* surface, uint32_t time,
                                        int touchId, int touchState, const Point& touchPos)
{
    TouchIDMap::iterator find_iter;
    bool allTouchPointsReleased = false;
    switch (touchState)
    {
    case WL_TOUCH_DOWN:
    	find_iter = m_touchIdMap.find(touchId);
    	if ((find_iter != m_touchIdMap.end()) && (WL_TOUCH_DOWN == find_iter->second))
    	{
            LOG_ERROR("WaylandInputDevice", "Multiple touch downs from device without Touch UP!!!, "
                      "time=" << time << ", touch ID=" << touchId << ", "
                      "touch position: x=" << touchPos.x << ", "
                      "y=" << touchPos.y);
    	}

    	m_touchIdMap[touchId] = WL_TOUCH_DOWN;
    	if (!m_wlTouchFocus)
    	{
            setTouchFocus(surface);

            LOG_DEBUG("WaylandInputDevice", "Touch down focus set, "
                      "native surface=" << surface << ", "
                      "time=" << time << ", touch ID=" << touchId << ", "
                      "touch state=" << touchState << ", "
                      "touch position: x=" << touchPos.x << ", "
                      "y=" << touchPos.y);
        }
        if (m_wlTouchFocusResource && m_wlTouchFocus)
        {
            wl_touch_send_down(m_wlTouchFocusResource,
                                0 /*serial*/, time,
                                &m_wlTouchFocus->resource, touchId,
                                wl_fixed_from_double(touchPos.x),
                                wl_fixed_from_double(touchPos.y));

            LOG_DEBUG("WaylandInputDevice", "Send touch down, "
                      "native surface=" << surface << ", "
                      "time=" << time << ", touch ID=" << touchId << ", "
                      "touch state=" << touchState << ", "
                      "touch position: x=" << touchPos.x << ", "
                      "y=" << touchPos.y);
        }
        break;
    case WL_TOUCH_MOTION:
        if (!m_wlTouchFocus)
        {
            LOG_DEBUG("WaylandInputDevice", "Touch motion focus not set, "
                      "native surface=" << surface << ", "
                      "time=" << time << ", touch ID=" << touchId << ", "
                      "touch state=" << touchState << ", "
                      "touch position: x=" << touchPos.x << ", "
                      "y=" << touchPos.y);
            break;
        }
        if (m_wlTouchFocusResource)
        {
            wl_touch_send_motion(m_wlTouchFocusResource,
                                 time, touchId,
                                 wl_fixed_from_double(touchPos.x),
                                 wl_fixed_from_double(touchPos.y));
            LOG_DEBUG("WaylandInputDevice", "Send touch motion, "
                      "native surface=" << surface << ", "
                      "time=" << time << ", touch ID=" << touchId << ", "
                      "touch state=" << touchState << ", "
                      "touch position: x=" << touchPos.x << ", "
                      "y=" << touchPos.y);
        }
        break;
    case WL_TOUCH_UP:
        /*set touch id state to touch up*/
        TouchIDMap::iterator iter = m_touchIdMap.begin();
        TouchIDMap::iterator end = m_touchIdMap.end();
        find_iter = m_touchIdMap.find(touchId);
    	if ((find_iter != m_touchIdMap.end()) && (WL_TOUCH_UP == find_iter->second))
    	{
            LOG_ERROR("WaylandInputDevice", "Multiple touch UPs from device without Touch DOWN!!!, "
                      "time=" << time << ", touch ID=" << touchId << ", "
                      "touch position: x=" << touchPos.x << ", "
                      "y=" << touchPos.y);
    	}

    	m_touchIdMap[touchId] = WL_TOUCH_UP;
        if (m_wlTouchFocusResource)
        {
            wl_touch_send_up(m_wlTouchFocusResource, 0 /*serial*/, time, touchId);

            LOG_DEBUG("WaylandInputDevice", "Send touch up, "
                      "native surface=" << surface << ", "
                      "time=" << time << ", touch ID=" << touchId << ", "
                      "touch state=" << touchState << ", "
                      "touch position: x=" << touchPos.x << ", "
                      "y=" << touchPos.y);
        }

        allTouchPointsReleased = true;
        for (; iter != end; ++iter)
        {
            if (iter->second != WL_TOUCH_UP)
            {
                allTouchPointsReleased = false;
            }
        }

        if(allTouchPointsReleased)
        {
            m_touchIdMap.clear();
            setTouchFocus(NULL);

            LOG_DEBUG("WaylandInputDevice", "Set touch focus to NULL, "
                      "native surface=" << surface << ", "
                      "time=" << time << ", touch ID=" << touchId << ", "
                      "touch state=" << touchState << ", "
                      "touch position: x=" << touchPos.x << ", "
                      "y=" << touchPos.y);
        }
        break;
    }
}

void
WaylandInputDevice::sendTouchFrameEvent()
{
    wl_touch *touch = touchDevice();
    wl_resource *resource = touch->focus_resource;
    if (resource)
    {
        wl_touch_send_frame(resource);

        LOG_DEBUG("WaylandInputDevice",
                  "touch=" << touch << ", "
                  "resource=" << resource);
    }
    else
    {
        LOG_DEBUG("WaylandInputDevice",
                  "Resource not set, "
                  "touch=" << touch);

    }
}

void
WaylandInputDevice::sendTouchCancelEvent()
{
    wl_touch *touch = touchDevice();
    wl_resource *resource = touch->focus_resource;
    if (resource)
    {
        wl_touch_send_cancel(resource);

        LOG_DEBUG("WaylandInputDevice",
                  "Resource=" << resource);
    }
    else
    {
        LOG_DEBUG("WaylandInputDevice",
                  "Resource not set, "
                  "touch=" << touch);
    }
}

void
WaylandInputDevice::setMouseFocus(struct wl_surface* surface,
                                    const Point& globalPos,
                                    const Point& localPos)
{

    LOG_DEBUG("WaylandInputDevice",
              "native surface=" << surface << ", "
              "global position: x=" << globalPos.x << ", y=" << globalPos.y
              << ", local position: x=" << localPos.x << ", y=" << localPos.y);

    wl_pointer* pointer = pointerDevice();
    pointer->x = wl_fixed_from_double(globalPos.x);
    pointer->y = wl_fixed_from_double(globalPos.y);
    pointer->current = surface;
    pointer->current_x = wl_fixed_from_double(localPos.x);
    pointer->current_y = wl_fixed_from_double(localPos.y);
    pointer->grab->interface->focus(pointer->grab,
                                    surface,
                                    wl_fixed_from_double(localPos.x),
                                    wl_fixed_from_double(localPos.y));
}

void
WaylandInputDevice::setKeyboardFocus(struct wl_surface* surface)
{
    LOG_DEBUG("WaylandInputDevice",
              "native surface=" << surface << ", "
              "keyboard=" << keyboardDevice());
    wl_keyboard_set_focus(keyboardDevice(), surface);
}

void
WaylandInputDevice::cleanupTouchFocus()
{
    if (!m_touchIdMap.empty())
    {
        LOG_WARNING("WaylandInputDevice", "Touch focus reset without Sending Touch UP event!!!!");
        m_touchIdMap.clear();
    }

    m_wlSeat.touch->focus = NULL;
    m_wlSeat.touch->focus_resource = NULL;
    m_wlTouchFocus = NULL;
    m_wlTouchFocusResource = NULL;
}

void
WaylandInputDevice::setTouchFocus(struct wl_surface* surface)
{
    struct wl_resource* resource = NULL;

    if (m_wlTouchFocus == surface)
        return;

    if (surface)
    {
        resource = findResourceForClient(&m_wlSeat.touch->resource_list,
                                        surface->resource.client);
        if (resource)
        {
            m_wlSeat.touch->focus = surface;
            m_wlSeat.touch->focus_resource = resource;
        }
        else
        {
            m_wlSeat.touch->focus = NULL;
            m_wlSeat.touch->focus_resource = NULL;
            m_touchIdMap.clear();
        }
    }
    else
    {
        m_wlSeat.touch->focus = NULL;
        m_wlSeat.touch->focus_resource = NULL;
        m_touchIdMap.clear();
    }

    LOG_DEBUG("WaylandInputDevice",
              "touch device=" << m_wlSeat.touch << ", "
              "native surface=" << surface << ", resource=" << resource);

    m_wlTouchFocus = m_wlSeat.touch->focus;
    m_wlTouchFocusResource = m_wlSeat.touch->focus_resource;
}

void
WaylandInputDevice::sendModifiers(uint32_t serial)
{
    struct wl_keyboard *keyboard = keyboardDevice();
    struct wl_keyboard_grab *grab = keyboard->grab;

    LOG_DEBUG("WaylandInputDevice",
              "keyboard=" << keyboard << ", "
              "serial=" << serial);

    grab->interface->modifiers(grab, serial,
                                keyboard->modifiers.mods_depressed,
                                keyboard->modifiers.mods_latched,
                                keyboard->modifiers.mods_locked,
                                keyboard->modifiers.group);
}


void
WaylandInputDevice::ResetAllFocuses(struct wl_surface* surface)
{
    /*Reset focus of all devices which are pointing to this
	 *surface*/
    if (hasPointer())
    {
        wl_pointer* pointer = pointerDevice();
        if ((NULL != pointer))
        {
            if (pointer->focus == surface)
            {
                pointer->current = NULL;
                if(pointer->button_count)
                {
                    LOG_WARNING("WaylandInputDevice",
                                "setting pointer focus surface as NULL, Before sending Release event ." );

               /* Resetting button count as zero here, so further
                * pending events from input handling thread for this
                * surface will be ignored in notification call back itself
                */
                    pointer->button_count = 0;
                }

                pointer->grab->interface->focus(pointer->grab, NULL,
                pointer->current_x, pointer->current_y);
            }
        }
    }
    if (hasKeyboard())
    {
        struct wl_keyboard* keyboard = keyboardDevice();
        if ((NULL != keyboard))
        {
            if (keyboard->focus == surface)
		         {
                setKeyboardFocus(NULL);
            }
        }
    }
    if (hasTouch())
    {
        if (m_wlTouchFocus == surface)
        {
            setTouchFocus(NULL);
        }
    }
}

